<?php
if (!defined('ABSPATH')) exit;

/**
 * Widget Elementor para páginas/plantillas de producto (WooCommerce).
 * Detecta el producto actual; opcionalmente permite forzar un ID.
 * Permite inline, controles y mostrar icono, reutilizando las vistas del plugin.
 */
class FP360_Elementor_WC_Product_Widget extends \Elementor\Widget_Base {

  public function get_name() { return 'fp360_wc_product_widget'; }
  public function get_title() { return 'Fotoproducto 360 — Producto Woo'; }
  public function get_icon() { return 'eicon-product-info'; }

  public function get_categories() {
    $cats = \Elementor\Plugin::$instance->elements_manager->get_categories();
    return isset($cats['woocommerce-elements']) ? ['woocommerce-elements'] : ['general'];
  }

  protected function register_controls() {
    $this->start_controls_section('content', ['label' => 'Contenido']);

    $this->add_control('product_id', [
      'label'       => 'ID de Producto (opcional)',
      'description' => 'Déjalo vacío para usar el producto actual.',
      'type'        => \Elementor\Controls_Manager::NUMBER,
      'min'         => 1,
      'step'        => 1,
      'default'     => 0,
    ]);

    $this->add_control('inline', [
      'label'        => 'Mostrar Inline (sin Modal)',
      'type'         => \Elementor\Controls_Manager::SWITCHER,
      'label_on'     => 'Sí',
      'label_off'    => 'No',
      'return_value' => '1',
      'default'      => '',
    ]);

    $this->add_control('controls', [
      'label'        => 'Controles (Play/Pause y Arrastre)',
      'type'         => \Elementor\Controls_Manager::SWITCHER,
      'label_on'     => 'Sí',
      'label_off'    => 'No',
      'return_value' => '1',
      'default'      => '',
    ]);

    $this->add_control('use_icon', [
      'label'        => 'Mostrar Icono en lugar de Texto',
      'type'         => \Elementor\Controls_Manager::SWITCHER,
      'label_on'     => 'Sí',
      'label_off'    => 'No',
      'return_value' => '1',
      'default'      => '',
      'condition'    => ['inline' => ''], // solo si es modal
    ]);

    $this->add_control('button', [
      'label'       => 'Texto del Botón (si no es inline)',
      'type'        => \Elementor\Controls_Manager::TEXT,
      'default'     => 'Ver en 360',
      'placeholder' => 'Ver en 360',
      'condition'   => ['inline' => ''],
    ]);

    $this->end_controls_section();
  }

  protected function render() {
    // resolver producto
    $pid = (int) $this->get_settings_for_display('product_id');
    $product = null;

    if ($pid > 0 && function_exists('wc_get_product')) {
      $product = wc_get_product($pid);
    } else {
      global $product;
      if (!$product instanceof \WC_Product && function_exists('wc_get_product')) {
        $qid = get_the_ID();
        if ($qid) $product = wc_get_product($qid);
      }
    }

    if (!$product instanceof \WC_Product) {
      if (\Elementor\Plugin::$instance->editor->is_edit_mode()) {
        echo '<div style="padding:8px;border:1px dashed #ccc;background:#fff;">Fotoproducto 360: selecciona una plantilla de producto o indica un ID.</div>';
      }
      return;
    }

    $post_id = $product->get_id();

    // metadatos del plugin (imágenes/fps)
    $ids = get_post_meta($post_id, \FP360\WC::META_IMAGES, true);
    if (!is_array($ids) || empty($ids)) {
      if (\Elementor\Plugin::$instance->editor->is_edit_mode()) {
        echo '<div style="padding:8px;border:1px dashed #ccc;background:#fff;">Fotoproducto 360: este producto no tiene imágenes 360.</div>';
      }
      return;
    }

    $fps = (int) get_post_meta($post_id, \FP360\WC::META_FPS, true);
    if ($fps <= 0) $fps = 24;

    $urls = [];
    foreach ($ids as $id) {
      $u = wp_get_attachment_image_url((int)$id, 'full');
      if ($u) $urls[] = $u;
    }
    if (empty($urls)) return;

    // assets frontend
    $assets_base = defined('FP360_ASSETS_URL') ? FP360_ASSETS_URL : plugin_dir_url(dirname(__FILE__)) . 'assets/';
    $version     = defined('FP360_VERSION') ? FP360_VERSION : '1.0.0';

    wp_enqueue_style('fp360-frontend', $assets_base . 'frontend/frontend.css', [], $version);
    wp_enqueue_script('fp360-frontend', $assets_base . 'frontend/frontend.js', [], $version, true);

    // ajustes del widget
    $inline   = $this->get_settings_for_display('inline') ? '1' : '0';
    $controls = $this->get_settings_for_display('controls') ? true : false;
    $use_icon = $this->get_settings_for_display('use_icon') ? true : false;
    $button   = (string) $this->get_settings_for_display('button');
    if (!$button) $button = 'Ver en 360';

    // datos para vistas
    $data = [
      'images'   => $urls,
      'fps'      => $fps,
      'title'    => get_the_title($post_id),
      'controls' => $controls,
    ];

    // render
    if ($inline === '1') {
      \FP360\Utils::view('inline-viewer.php', ['data' => $data]);
    } else {
      $icon_url = $assets_base . 'frontend/360.webp';
      \FP360\Utils::view('product-button.php', [
        'button_text' => $button,
        'data'        => $data,
        'use_icon'    => $use_icon,
        'icon_url'    => $icon_url,
      ]);
      
      // ✅ NUEVO ENFOQUE: Imprimir el modal directamente si aún no existe
      // En lugar de usar wp_footer (que puede ser demasiado tarde), lo imprimimos directamente
      $this->ensure_modal_exists();
    }
  }

  /**
   * Asegura que el modal existe en el DOM para Elementor
   */
  private function ensure_modal_exists() {
    static $modal_added = false;
    if ($modal_added) return;
    $modal_added = true;
    
    // Usar output buffering para capturar y luego imprimir el modal
    ob_start();
    \FP360\WC::print_shared_modal();
    $modal_html = ob_get_clean();
    
    // Imprimir el modal directamente
    echo $modal_html;
  }
}
