<?php
namespace FP360;

if (!defined('ABSPATH')) exit;

class WC {
  const META_IMAGES   = '_fp360_images';
  const META_FPS      = '_fp360_fps';
  const META_CONTROLS = '_fp360_controls'; // bool mostrar controles
  const META_ICON     = '_fp360_icon';     // bool mostrar icono en botón

  public static function register() {
    if (!self::is_wc_active()) return;

    add_action('add_meta_boxes', [__CLASS__, 'add_meta_box']);
    add_action('save_post_product', [__CLASS__, 'save_meta']);
    add_action('admin_enqueue_scripts', [__CLASS__, 'admin_assets']);

    add_action('wp', [__CLASS__, 'hook_front_render']);
    add_action('wp_enqueue_scripts', [__CLASS__, 'maybe_enqueue_front_assets']);
    
    // AJAX para ordenar por nombre
    add_action('wp_ajax_fp360_get_attachment_names', [__CLASS__, 'ajax_get_attachment_names']);
  }

  public static function is_wc_active(): bool { return class_exists('\\WooCommerce'); }

  public static function add_meta_box() {
    add_meta_box('fp360_wc_box', 'Photo 360', [__CLASS__, 'mb_render'], 'product', 'normal', 'high');
  }

  public static function mb_render(\WP_Post $post) {
    wp_nonce_field('fp360_wc_save', 'fp360_wc_nonce');

    $ids = get_post_meta($post->ID, self::META_IMAGES, true);
    if (!is_array($ids)) $ids = [];
    $thumbs = array_map('intval', $ids);

    $fps      = (int) get_post_meta($post->ID, self::META_FPS, true);
    if ($fps <= 0) $fps = 24;
    $controls = (bool) get_post_meta($post->ID, self::META_CONTROLS, true);
    $useIcon  = (bool) get_post_meta($post->ID, self::META_ICON, true);

    echo '<div id="fp360p-box">';
    echo '<input type="hidden" id="fp360p-ids" name="fp360p_ids" value="' . esc_attr(implode(',', $thumbs)) . '" />';
    echo '<p><button type="button" class="button" id="fp360p-add">Añadir imágenes</button> ';
    echo '<button type="button" class="button" id="fp360p-invert">Invertir orden</button> ';
    echo '<button type="button" class="button" id="fp360p-sort-name">Ordenar por nombre</button></p>';

    echo '<ul id="fp360p-list" class="fp360-sortable">';
    foreach ($thumbs as $id) {
      $src = wp_get_attachment_image_url($id, 'thumbnail');
      echo '<li data-id="' . esc_attr($id) . '"><img src="' . esc_url($src) . '"><span class="fp360-remove">x</span></li>';
    }
    echo '</ul>';
    echo '<p class="description">Arrastra para ordenar; el orden va de izquierda a derecha.</p>';

    echo '<p><label for="fp360p-fps">Fotogramas por segundo</label> ';
    echo '<input type="number" min="1" max="120" id="fp360p-fps" name="fp360p_fps" value="' . esc_attr($fps) . '" class="small-text" /></p>';

    echo '<p><label><input type="checkbox" name="fp360p_controls" value="1" ' . checked($controls, true, false) . ' /> ';
    echo 'Mostrar controles (play/pause y drag)</label></p>';

    echo '<p><label><input type="checkbox" name="fp360p_icon" value="1" ' . checked($useIcon, true, false) . ' /> ';
    echo 'Mostrar icono en el botón (en lugar de texto)</label></p>';

    echo '</div>';
  }

  public static function save_meta($post_id) {
    if (!isset($_POST['fp360_wc_nonce']) || !wp_verify_nonce($_POST['fp360_wc_nonce'], 'fp360_wc_save')) return;
    if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) return;
    if (!current_user_can('edit_post', $post_id)) return;

    $raw = isset($_POST['fp360p_ids']) ? (string) $_POST['fp360p_ids'] : '';
    $ids = array_filter(array_map('intval', explode(',', $raw)));
    update_post_meta($post_id, self::META_IMAGES, $ids);

    $fps = isset($_POST['fp360p_fps']) ? (int) $_POST['fp360p_fps'] : 24;
    $fps = max(1, min(120, $fps));
    update_post_meta($post_id, self::META_FPS, $fps);

    $controls = !empty($_POST['fp360p_controls']);
    $icon     = !empty($_POST['fp360p_icon']);

    $controls ? update_post_meta($post_id, self::META_CONTROLS, 1) : delete_post_meta($post_id, self::META_CONTROLS);
    $icon     ? update_post_meta($post_id, self::META_ICON, 1)     : delete_post_meta($post_id, self::META_ICON);
  }

  public static function admin_assets($hook) {
    global $post_type;
    if ($post_type !== 'product') return;
    wp_enqueue_media();
    wp_enqueue_script('jquery-ui-sortable');
    wp_enqueue_style('fp360-admin', FP360_ASSETS_URL . 'admin/admin.css', [], FP360_VERSION);
    wp_enqueue_script('fp360-admin', FP360_ASSETS_URL . 'admin/admin.js', ['jquery', 'jquery-ui-sortable'], FP360_VERSION, true);
    
    // Localizar script para AJAX
    wp_localize_script('fp360-admin', 'fp360_ajax', [
      'ajaxurl' => admin_url('admin-ajax.php'),
      'nonce' => wp_create_nonce('fp360_wc_save')
    ]);
  }

  public static function hook_front_render() {
    if (!is_product()) return;
    add_action('woocommerce_single_product_summary', [__CLASS__, 'render_button'], 25);
  }

  public static function render_button() {
    if (!is_product()) return;
    global $product;
    if (!$product instanceof \WC_Product) return;

    $post_id = $product->get_id();
    $ids = get_post_meta($post_id, self::META_IMAGES, true);
    if (!is_array($ids) || empty($ids)) return;

    $fps = (int) get_post_meta($post_id, self::META_FPS, true);
    if ($fps <= 0) $fps = 24;

    $urls = [];
    foreach ($ids as $id) {
      $u = wp_get_attachment_image_url((int)$id, 'full');
      if ($u) $urls[] = $u;
    }
    if (empty($urls)) return;

    // assets front
    wp_enqueue_style('fp360-frontend', FP360_ASSETS_URL . 'frontend/frontend.css', [], FP360_VERSION);
    wp_enqueue_script('fp360-frontend', FP360_ASSETS_URL . 'frontend/frontend.js', [], FP360_VERSION, true);

    // agregar el modal al footer solo cuando hay contenido 360
    add_action('wp_footer', [__CLASS__, 'print_shared_modal']);

    // controles desde meta (permitimos override por filtro)
    $controls_meta = (bool) get_post_meta($post_id, self::META_CONTROLS, true);
    $controls = (bool) apply_filters('fp360_wc_controls', $controls_meta, $product, $post_id);

    // icono desde meta (permite filtro si quieres)
    $icon_meta = (bool) get_post_meta($post_id, self::META_ICON, true);
    $use_icon  = (bool) apply_filters('fp360_wc_use_icon', $icon_meta, $product, $post_id);

    $data = [
      'images'   => $urls,
      'fps'      => $fps,
      'title'    => get_the_title($post_id),
      'controls' => $controls,
    ];

    $assets_base = defined('FP360_ASSETS_URL') ? FP360_ASSETS_URL : plugin_dir_url(dirname(__FILE__)) . 'assets/';
    $icon_url = $assets_base . 'frontend/360.webp';

    Utils::view('product-button.php', [
      'button_text' => 'ver 360',
      'data'        => $data,
      'use_icon'    => $use_icon,
      'icon_url'    => $icon_url,
    ]);
  }

  public static function maybe_enqueue_front_assets() {
    if (!is_product()) return;
    global $product;
    if (!$product instanceof \WC_Product) return;
    $ids = get_post_meta($product->get_id(), self::META_IMAGES, true);
    if (is_array($ids) && !empty($ids)) {
      wp_enqueue_style('fp360-frontend', FP360_ASSETS_URL . 'frontend/frontend.css', [], FP360_VERSION);
      wp_enqueue_script('fp360-frontend', FP360_ASSETS_URL . 'frontend/frontend.js', [], FP360_VERSION, true);
    }
  }

  /**
   * Maneja la petición AJAX para obtener nombres de archivos adjuntos
   */
  public static function ajax_get_attachment_names() {
    // verificar permisos y nonce
    if (!current_user_can('edit_posts')) {
      wp_die('Sin permisos');
    }
    
    if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'fp360_wc_save')) {
      wp_die('Nonce inválido');
    }
    
    $ids = isset($_POST['ids']) ? array_map('intval', (array) $_POST['ids']) : [];
    
    if (empty($ids)) {
      wp_send_json_error('No hay IDs válidos');
    }
    
    $names = [];
    
    foreach ($ids as $id) {
      $attachment = get_post($id);
      if ($attachment && $attachment->post_type === 'attachment') {
        // obtener el nombre del archivo (sin la ruta)
        $file_path = get_attached_file($id);
        if ($file_path) {
          $names[$id] = basename($file_path);
        } else {
          // fallback al post_title si no hay archivo
          $names[$id] = $attachment->post_title ?: 'imagen_' . $id;
        }
      } else {
        $names[$id] = 'imagen_' . $id; // fallback
      }
    }
    
    wp_send_json_success($names);
  }

  /**
   * Imprime el modal compartido para productos WooCommerce
   */
  public static function print_shared_modal() {
    if (is_admin()) return;
    // Evitar duplicar el modal si ya se imprimió por galerías
    static $modal_printed = false;
    if ($modal_printed) return;
    $modal_printed = true;
?>
    <div id="fp360-modal" class="fp360-modal" aria-hidden="true">
      <div class="fp360-modal-backdrop" data-fp360-close></div>
      <div class="fp360-modal-dialog" role="dialog" aria-modal="true" aria-label="visor 360">
        <button class="fp360-modal-close" type="button" data-fp360-close>×</button>
        <div class="fp360-controls" aria-hidden="true">
          <button type="button" class="fp360-ctrl fp360-playpause" data-mode="pause" aria-label="play/pause"></button>
        </div>
        <canvas id="fp360-canvas"></canvas>
      </div>
    </div>
<?php
  }
}
