<?php
namespace FP360;

if (!defined('ABSPATH')) exit;

class License {
  // abre el gate si el estado está ok o si ping dice ok
  public static function maybe_open_gate(): bool {
    // lee cache
    $state = get_site_transient(FP360_TRANS_LICENSE_STATE);
    if (is_array($state)) {
      if (!empty($state['active']) && !empty($state['until']) && $state['until'] > Utils::now()) {
        return true;
      }
    }

    // si no hay cache o caducó, intentar ping
    return self::ping_and_cache();
  }

  // activa una licencia llamando a activate-simple
  public static function activate($license_key): array {
    $key  = Utils::clean_key($license_key);
    if (!$key) return ['ok' => false, 'msg' => 'clave no válida'];

    $payload = [
      'license_key' => $key,
      'app_slug'    => 'producto-360-wp-plugin',
      'site'        => home_url(),
    ];

    $res = Utils::http_post('licenses/activate', $payload, 10);
    if (is_wp_error($res)) return ['ok' => false, 'msg' => $res->get_error_message()];
    if (($res['code'] ?? 0) !== 200) return ['ok' => false, 'msg' => 'error activando la licencia'];

    $data = $res['json']['data'] ?? [];
    $secret = $data['license_secret'] ?? '';
    $domain = $data['domain'] ?? '';
    if (!$secret) return ['ok' => false, 'msg' => 'respuesta sin secreto'];

    // guarda en options
    update_option(FP360_OPT_LICENSE_KEY, $key, false);
    update_option(FP360_OPT_LICENSE_SECRET, $secret, false);
    if ($domain) update_option(FP360_OPT_LICENSE_DOMAIN, $domain, false);

    // cachea estado inicial con pequeña gracia
    $ttl = intval($data['grace_seconds'] ?? 1800);
    set_site_transient(FP360_TRANS_LICENSE_STATE, [
      'active' => true,
      'until'  => Utils::now() + max(60, $ttl),
    ], max(60, $ttl));

    return ['ok' => true, 'msg' => 'licencia activada'];
  }

  // desactiva localmente (no llama backend)
  public static function deactivate_local() {
    delete_option(FP360_OPT_LICENSE_KEY);
    delete_option(FP360_OPT_LICENSE_SECRET);
    delete_option(FP360_OPT_LICENSE_DOMAIN);
    delete_site_transient(FP360_TRANS_LICENSE_STATE);
  }

  // hace ping firmado al backend y cachea el resultado
  public static function ping_and_cache(): bool {
    $key    = get_option(FP360_OPT_LICENSE_KEY);
    $secret = get_option(FP360_OPT_LICENSE_SECRET);
    if (!$key || !$secret) {
      delete_site_transient(FP360_TRANS_LICENSE_STATE);
      return false;
    }

    $slug  = 'producto-360-wp-plugin';
    $site  = home_url();
    $host  = Utils::host();
    $ts    = Utils::now();
    $nonce = wp_generate_password(8, false);

    // firma base64url(hmac_sha256(payload, secret))
    $payload = "{$slug}|{$host}|{$ts}|{$nonce}";
    $raw_sig = hash_hmac('sha256', $payload, $secret, true);
    $sig     = rtrim(strtr(base64_encode($raw_sig), '+/', '-_'), '=');

    $res = Utils::http_post('licenses/ping', [
      'license_key' => $key,
      'app_slug'    => $slug,
      'site'        => $site,
      'ts'          => $ts,
      'nonce'       => $nonce,
      'sig'         => $sig,
    ], 8);

    if (is_wp_error($res)) {
      // si falla la red, no abrimos gate
      delete_site_transient(FP360_TRANS_LICENSE_STATE);
      return false;
    }

    $ok   = ($res['code'] ?? 0) === 200;
    $data = $res['json']['data'] ?? [];
    $active = $ok && !empty($data['active']);

    if ($active) {
      $next = intval($data['next_ping_in'] ?? 1800);
      set_site_transient(FP360_TRANS_LICENSE_STATE, [
        'active' => true,
        'until'  => Utils::now() + max(60, $next),
      ], max(60, $next));
      return true;
    }

    delete_site_transient(FP360_TRANS_LICENSE_STATE);
    return false;
  }

  // helpers para admin
  public static function is_active_cached(): bool {
    $state = get_site_transient(FP360_TRANS_LICENSE_STATE);
    return is_array($state) && !empty($state['active']) && $state['until'] > Utils::now();
  }

  public static function stored_key(): string {
    return (string) get_option(FP360_OPT_LICENSE_KEY, '');
  }





  public static function handle_reset_action() {
    if (!current_user_can('manage_options')) wp_die('forbidden');
    check_admin_referer('fp360_reset_license');

    // borra datos locales (usa tu método actual)
    self::deactivate_local();

    // borra cron si lo usas (opcional)
    if (function_exists('wp_next_scheduled')) {
      $ts = wp_next_scheduled('fp360_license_ping');
      if ($ts) wp_unschedule_event($ts, 'fp360_license_ping');
    }

    // redirige de vuelta a la página de licencia con un flag
    wp_safe_redirect(add_query_arg(['page' => 'fp360-license', 'reset' => '1'], admin_url('admin.php')));
    exit;
  }

  


}
if (is_admin()) {
  add_action('admin_post_fp360_reset_license', ['FP360\\License', 'handle_reset_action']);
}