<?php

namespace FP360;

if (!defined('ABSPATH')) exit;

class Galleries
{
    const CPT = 'fp360_gallery';
    const META_IMAGES = '_fp360_gallery_images';
    const META_FPS    = '_fp360_gallery_fps';

    public static function register()
    {
        add_action('init', [__CLASS__, 'register_cpt']);
        add_action('add_meta_boxes', [__CLASS__, 'add_meta_boxes']);
        add_action('save_post_' . self::CPT, [__CLASS__, 'save_meta']);
        add_action('admin_enqueue_scripts', [__CLASS__, 'admin_assets']);
        add_shortcode('fp360_gallery', [__CLASS__, 'shortcode']);

        // docs en el admin
        add_action('admin_menu', [__CLASS__, 'register_docs_page']);

        // columnas en la tabla del CPT
        add_filter('manage_' . self::CPT . '_posts_columns', [__CLASS__, 'columns']);
        add_action('manage_' . self::CPT . '_posts_custom_column', [__CLASS__, 'column_content'], 10, 2);
        add_filter('manage_edit-' . self::CPT . '_sortable_columns', [__CLASS__, 'sortable_columns']);

        // assets frontend solo cuando hay shortcode en la página
        add_action('wp_enqueue_scripts', [__CLASS__, 'maybe_enqueue_front_assets']);
        
        // AJAX para ordenar por nombre en galerías
        add_action('wp_ajax_fp360_get_attachment_names', [__CLASS__, 'ajax_get_attachment_names']);
    }

    public static function register_cpt()
    {
        $labels = [
            'name' => 'Galerías 360',
            'singular_name' => 'Galería 360',
            'add_new' => 'Añadir nueva',
            'add_new_item' => 'Añadir galería',
            'edit_item' => 'Editar galería',
            'new_item' => 'Nueva galería',
            'view_item' => 'Ver galería',
            'search_items' => 'Buscar galerías',
            'not_found' => 'No hay galerías',
            'menu_name' => 'Galerías',
        ];
        register_post_type(self::CPT, [
            'labels' => $labels,
            'public' => false,
            'show_ui' => true,
            'show_in_menu' => 'fp360', // cuelga del menú del plugin
            'supports' => ['title'],
            'menu_position' => null,
            'capability_type' => 'post',
        ]);
    }

    public static function add_meta_boxes()
    {
        add_meta_box('fp360_gallery_images', 'Imágenes 360', [__CLASS__, 'mb_images'], self::CPT, 'normal', 'high');
        add_meta_box('fp360_gallery_fps', 'FPS', [__CLASS__, 'mb_fps'], self::CPT, 'side', 'default');

        // metabox lateral con ID y shortcode
        add_meta_box('fp360_gallery_info', 'Información', [__CLASS__, 'mb_info'], self::CPT, 'side', 'high');
    }

    // metabox imágenes
    public static function mb_images($post)
    {
        wp_nonce_field('fp360_gallery_save', 'fp360_gallery_nonce');
        $ids = get_post_meta($post->ID, self::META_IMAGES, true);
        if (!is_array($ids)) $ids = [];
        $thumbs = array_map('intval', $ids);
        echo '<div id="fp360-gallery-box">';
        echo '<input type="hidden" id="fp360-gallery-ids" name="fp360_gallery_ids" value="' . esc_attr(implode(',', $thumbs)) . '" />';
        echo '<p><button type="button" class="button" id="fp360-gallery-add">Añadir imágenes</button> ';
        echo '<button type="button" class="button" id="fp360-gallery-invert">Invertir orden</button> ';
        echo '<button type="button" class="button" id="fp360-gallery-sort-name">Ordenar por nombre</button></p>';
        echo '<ul id="fp360-gallery-list" class="fp360-sortable">';
        foreach ($thumbs as $id) {
            $src = wp_get_attachment_image_url($id, 'thumbnail');
            echo '<li data-id="' . esc_attr($id) . '"><img src="' . esc_url($src) . '" /><span class="fp360-remove">x</span></li>';
        }
        echo '</ul>';
        echo '<p class="description">Arrastra para ordenar; se usará el orden de izquierda a derecha.</p>';
        echo '</div>';
    }

    // metabox fps
    public static function mb_fps($post)
    {
        $fps = (int) get_post_meta($post->ID, self::META_FPS, true);
        if ($fps <= 0) $fps = 24;
        echo '<label for="fp360_gallery_fps">Fotogramas por segundo</label>';
        echo '<input type="number" min="1" max="120" id="fp360_gallery_fps" name="fp360_gallery_fps" value="' . esc_attr($fps) . '" class="small-text" />';
    }

    // metabox info (ID y shortcode)
    public static function mb_info($post)
    {
        $gid = (int) $post->ID;
        $short = '[fp360_gallery id="' . $gid . '"]';
        echo '<p><strong>ID:</strong> ' . esc_html($gid) . '</p>';
        echo '<p><label for="fp360_gallery_shortcode"><strong>Shortcode</strong></label></p>';
        echo '<p>' . esc_attr($short) . '</p>';
        echo '<p class="description">Copia y pega este shortcode donde quieras mostrar la galería.</p>';
    }

    public static function save_meta($post_id)
    {
        if (!isset($_POST['fp360_gallery_nonce']) || !wp_verify_nonce($_POST['fp360_gallery_nonce'], 'fp360_gallery_save')) return;
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) return;
        if (!current_user_can('edit_post', $post_id)) return;

        $raw = isset($_POST['fp360_gallery_ids']) ? (string) $_POST['fp360_gallery_ids'] : '';
        $ids = array_filter(array_map('intval', explode(',', $raw)));
        update_post_meta($post_id, self::META_IMAGES, $ids);

        $fps = isset($_POST['fp360_gallery_fps']) ? (int) $_POST['fp360_gallery_fps'] : 24;
        if ($fps < 1) $fps = 1;
        if ($fps > 120) $fps = 120;
        update_post_meta($post_id, self::META_FPS, $fps);
    }

    public static function admin_assets($hook)
    {
        global $post_type;
        if ($post_type !== self::CPT) return;

        wp_enqueue_media();
        wp_enqueue_script('jquery-ui-sortable');
        wp_enqueue_style('fp360-admin', FP360_ASSETS_URL . 'admin/admin.css', [], FP360_VERSION);
        wp_enqueue_script('fp360-admin', FP360_ASSETS_URL . 'admin/admin.js', ['jquery', 'jquery-ui-sortable'], FP360_VERSION, true);
        
        // Localizar script para AJAX en galerías
        wp_localize_script('fp360-admin', 'fp360_ajax', [
            'ajaxurl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('fp360_gallery_save')
        ]);
    }

    // ---------- DOCUMENTACIÓN EN ADMIN ----------
    public static function register_docs_page()
    {
        add_submenu_page(
            'fp360',
            'Documentación',
            'Documentación',
            'manage_options',
            'fp360-docs',
            [__CLASS__, 'render_docs_page']
        );
    }

    public static function render_docs_page()
    {
        if (!current_user_can('manage_options')) return;

        // css minimal aislado
        echo '<div class="wrap fp360-docs">';
        echo '<style>
      .fp360-docs h1{margin:12px 0 14px}
      .fp360-docs .card{background:#fff;border:1px solid #dcdcde;border-radius:8px;padding:16px;margin-bottom:16px}
      .fp360-docs table{width:100%;border-collapse:collapse}
      .fp360-docs th,.fp360-docs td{border-bottom:1px solid #eef0f2;padding:8px 10px;text-align:left;vertical-align:top}
      .fp360-docs code{background:#f6f7f7;padding:.15em .4em;border-radius:4px}
      .fp360-docs .muted{color:#50575e;font-size:12px}
      .fp360-docs .nowrap{white-space:nowrap}
      .fp360-docs .example{background:#f6f7f7;border:1px solid #e1e4e8;border-radius:6px;padding:10px;font-family:monospace;margin:10px 0}
    </style>';

        echo '<h1>Documentación De Shortcodes</h1>';

        // Shortcode principal: fp360_gallery
        echo '<div class="card">';
        echo '<h2><code>[fp360_gallery]</code></h2>';
        echo '<p class="muted">Inserta un visor 360 basado en una <strong>Galería 360</strong> creada en este plugin.</p>';

        echo '<h3>Parámetros</h3>';
        echo '<table>
      <thead><tr>
        <th class="nowrap">Parámetro</th>
        <th>Tipo</th>
        <th>Por Defecto</th>
        <th>Descripción</th>
      </tr></thead>
      <tbody>
        <tr>
          <td><code>id</code></td>
          <td>Entero</td>
          <td><em>Obligatorio</em></td>
          <td>ID de la galería 360 a renderizar.</td>
        </tr>
        <tr>
          <td><code>button</code></td>
          <td>Texto</td>
          <td><code>Ver en 360</code></td>
          <td>Texto del botón que abre el modal. Solo aplica si <code>inline="0"</code>.</td>
        </tr>
        <tr>
          <td><code>controls</code></td>
          <td>Booleano (0/1)</td>
          <td><code>0</code></td>
          <td>Muestra controles de <em>Play/Pause</em> y habilita arrastre. Si está en <code>0</code>, no hay controles ni drag.</td>
        </tr>
        <tr>
          <td><code>inline</code></td>
          <td>Booleano (0/1)</td>
          <td><code>0</code></td>
          <td>Si vale <code>1</code>, renderiza el visor embebido en la página sin modal ni botón.</td>
        </tr>
        <tr>
          <td><code>icon</code></td>
          <td>Booleano (0/1)</td>
          <td><code>0</code></td>
          <td>Si vale <code>1</code>, el botón mostrará un icono en lugar de texto (solo aplica cuando <code>inline="0"</code>).</td>
        </tr>
      </tbody>
    </table>';

        echo '<h3>Ejemplos</h3>';
        echo '<div class="example">[fp360_gallery id="123"]</div>';
        echo '<div class="example">[fp360_gallery id="123" controls="1" button="Ver en 360"]</div>';
        echo '<div class="example">[fp360_gallery id="123" icon="1"]</div>';
        echo '<div class="example">[fp360_gallery id="123" inline="1" controls="1"]</div>';

        echo '</div>';

        // Nota WooCommerce
        echo '<div class="card">';
        echo '<h2>WooCommerce</h2>';
        echo '<p class="muted">En productos de WooCommerce no se usa shortcode. Si un producto tiene imágenes 360 configuradas, el plugin añade automáticamente el botón de visualización en la ficha del producto.</p>';

        echo '<h3>Cómo configurarlo por producto</h3>';
        echo '<ol>
        <li>Ve a <strong>Productos → Editar</strong> el producto.</li>
        <li>En el metabox <strong>Photo 360</strong>:
          <ul>
            <li><strong>Imágenes 360</strong>: sube y ordena las imágenes (arrastra para reordenar).</li>
            <li><strong>FPS</strong>: ajusta los fotogramas por segundo.</li>
            <li><strong>Controles (Play/Pause y Arrastre)</strong>: activa si quieres interacción.</li>
            <li><strong>Mostrar Icono en el Botón</strong>: activa para usar el icono 360 en lugar del texto.</li>
          </ul>
        </li>
        <li>Guarda el producto.</li>
      </ol>';

        echo '<p class="muted">El botón aparecerá de forma automática en la página del producto. El estilo es neutro y el icono se muestra con un tamaño compacto adecuado para la mayoría de temas.</p>';
        echo '</div>';
    }


    // columnas de la tabla del CPT
    public static function columns($cols)
    {
        // inserta la columna ID tras el título
        $new = [];
        foreach ($cols as $key => $label) {
            $new[$key] = $label;
            if ($key === 'title') {
                $new['fp360_id'] = 'ID';
            }
        }
        if (!isset($new['fp360_id'])) $new['fp360_id'] = 'ID';
        return $new;
    }

    public static function column_content($col, $post_id)
    {
        if ($col === 'fp360_id') {
            echo (int) $post_id;
        }
    }

    public static function sortable_columns($cols)
    {
        // permite ordenar por ID
        $cols['fp360_id'] = 'ID';
        return $cols;
    }

    // ---------- SHORTCODE ----------
    // [fp360_gallery id="123" button="ver 360" controls="1" inline="1" icon="1"]
    // ---------- SHORTCODE ----------
    // [fp360_gallery id="123" button="ver 360" controls="1" inline="0" icon="1"]
    public static function shortcode($atts = [])
    {
        $atts = shortcode_atts([
            'id'       => 0,
            'button'   => 'ver 360',
            'controls' => '0',
            'inline'   => '0',
            'icon'     => '0', // '1' para mostrar icono 360.webp en el botón
        ], $atts, 'fp360_gallery');

        $post_id = (int) $atts['id'];
        if ($post_id <= 0) return '';

        $ids = get_post_meta($post_id, self::META_IMAGES, true);
        if (!is_array($ids) || empty($ids)) return '';

        $fps = (int) get_post_meta($post_id, self::META_FPS, true);
        if ($fps <= 0) $fps = 24;

        $urls = [];
        foreach ($ids as $id) {
            $u = wp_get_attachment_image_url((int)$id, 'full');
            if ($u) $urls[] = $u;
        }
        if (empty($urls)) return '';

        // assets front
        wp_enqueue_style('fp360-frontend', FP360_ASSETS_URL . 'frontend/frontend.css', [], FP360_VERSION);
        wp_enqueue_script('fp360-frontend', FP360_ASSETS_URL . 'frontend/frontend.js', [], FP360_VERSION, true);

        // datos del visor
        $data = [
            'images'   => $urls,
            'fps'      => $fps,
            'title'    => get_the_title($post_id),
            'controls' => ($atts['controls'] === '1'),
        ];

        // url del icono 360.webp en assets/frontend
        $assets_base = defined('FP360_ASSETS_URL') ? FP360_ASSETS_URL : plugin_dir_url(dirname(__FILE__)) . 'assets/';
        $icon_url = $assets_base . 'frontend/360.webp';

        ob_start();
        if ($atts['inline'] === '1') {
            // inline no usa botón (el flag icon no aplica)
            Utils::view('inline-viewer.php', ['data' => $data]);
        } else {
            Utils::view('gallery-button.php', [
                'button_text' => $atts['button'],
                'data'        => $data,
                'use_icon'    => ($atts['icon'] === '1'),
                'icon_url'    => $icon_url,
            ]);
        }
        return ob_get_clean();
    }

    public static function maybe_enqueue_front_assets()
    {
        if (is_admin()) return;
        if (is_singular()) {
            global $post;
            if ($post && has_shortcode($post->post_content, 'fp360_gallery')) {
                wp_enqueue_style('fp360-frontend', FP360_ASSETS_URL . 'frontend/frontend.css', [], FP360_VERSION);
                wp_enqueue_script('fp360-frontend', FP360_ASSETS_URL . 'frontend/frontend.js', [], FP360_VERSION, true);
                add_action('wp_footer', [__CLASS__, 'print_shared_modal']);
            }
        }
    }

    public static function print_shared_modal()
    {
        if (is_admin()) return;
        // Evitar duplicar el modal si ya se imprimió por WooCommerce
        static $modal_printed = false;
        if ($modal_printed) return;
        $modal_printed = true;
?>
        <div id="fp360-modal" class="fp360-modal" aria-hidden="true">
            <div class="fp360-modal-backdrop" data-fp360-close></div>
            <div class="fp360-modal-dialog" role="dialog" aria-modal="true" aria-label="visor 360">
                <button class="fp360-modal-close" type="button" data-fp360-close>×</button>
                <div class="fp360-controls" aria-hidden="true">
                    <button type="button" class="fp360-ctrl fp360-playpause" data-mode="pause" aria-label="play/pause"></button>
                </div>
                <canvas id="fp360-canvas"></canvas>
            </div>
        </div>
<?php
    }

    /**
     * Maneja la petición AJAX para obtener nombres de archivos adjuntos en galerías
     */
    public static function ajax_get_attachment_names() {
        // verificar permisos y nonce
        if (!current_user_can('edit_posts')) {
            wp_die('Sin permisos');
        }
        
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'fp360_gallery_save')) {
            wp_die('Nonce inválido');
        }
        
        $ids = isset($_POST['ids']) ? array_map('intval', (array) $_POST['ids']) : [];
        
        if (empty($ids)) {
            wp_send_json_error('No hay IDs válidos');
        }
        
        $names = [];
        
        foreach ($ids as $id) {
            $attachment = get_post($id);
            if ($attachment && $attachment->post_type === 'attachment') {
                // obtener el nombre del archivo (sin la ruta)
                $file_path = get_attached_file($id);
                if ($file_path) {
                    $names[$id] = basename($file_path);
                } else {
                    // fallback al post_title si no hay archivo
                    $names[$id] = $attachment->post_title ?: 'imagen_' . $id;
                }
            } else {
                $names[$id] = 'imagen_' . $id; // fallback
            }
        }
        
        wp_send_json_success($names);
    }
}
