(function(){
  // svg en base64 para iconos
  var ICONS = {
    play:  'url("data:image/svg+xml;utf8,<svg xmlns=\'http://www.w3.org/2000/svg\' viewBox=\'0 0 24 24\'><polygon fill=\'%23000\' points=\'8,5 19,12 8,19\'/></svg>")',
    pause: 'url("data:image/svg+xml;utf8,<svg xmlns=\'http://www.w3.org/2000/svg\' viewBox=\'0 0 24 24\'><rect x=\'6\' y=\'5\' width=\'4\' height=\'14\' fill=\'%23000\'/><rect x=\'14\' y=\'5\' width=\'4\' height=\'14\' fill=\'%23000\'/></svg>")'
  };
  var DPR = window.devicePixelRatio || 1;

  // viewer por instancia (sin globals)
  function createViewer(canvas, opts){
    var ctx = canvas.getContext('2d');
    var imgs = [], n = 0;
    var pos = 0;
    var playing = true;
    var fps = Math.max(1, opts.fps || 24);
    var timer = null;
    var dragging = false, lastX = null, wasPlayingBeforeDrag = false;
    var sensitivity = 0.5;

    function fit(){
      var rect = canvas.getBoundingClientRect();
      var w = rect.width > 0 ? rect.width : 600;
      var h = rect.height > 0 ? rect.height : 400;
      canvas.width = Math.round(w * DPR);
      canvas.height = Math.round(h * DPR);
      ctx.setTransform(DPR,0,0,DPR,0,0);
      if (n > 0) {
        var framesPerCanvas = n * 0.8;
        sensitivity = framesPerCanvas / Math.max(1, w);
      }
    }

    function draw(){
      if (!n) return;
      var p = pos % n; if (p < 0) p += n;
      var frame = Math.floor(p);
      var img = imgs[frame];
      if (!img) return;

      var w = canvas.width / DPR, h = canvas.height / DPR;
      var iw = img.naturalWidth || img.width, ih = img.naturalHeight || img.height;
      var scale = Math.min(w/iw, h/ih);
      var dw = iw * scale, dh = ih * scale;
      var dx = (w - dw)/2, dy = (h - dh)/2;
      ctx.clearRect(0,0,canvas.width,canvas.height);
      ctx.drawImage(img, dx, dy, dw, dh);
    }

    function tick(){ pos += 1; draw(); }

    function play(){
      if (!timer){
        timer = setInterval(tick, Math.max(10, Math.round(1000/ fps)));
        playing = true;
      }
    }
    function pause(){
      if (timer){
        clearInterval(timer);
        timer = null;
      }
      playing = false;
    }
    function toggle(){ playing ? pause() : play(); }

    function onDown(x){
      if (!opts.draggable) return;
      dragging = true;
      lastX = x;
      wasPlayingBeforeDrag = playing;
      pause();
    }
    function onMove(x){
      if (!dragging || !opts.draggable || !n) return;
      var dx = x - lastX; lastX = x;
      pos += dx * sensitivity;
      draw();
    }
    function onUp(){
      if (!dragging) return;
      dragging = false;
      if (opts.controls === true && wasPlayingBeforeDrag) {
        play();
      }
      if (opts.controls !== true) {
        play();
      }
    }

    function bindDrag(el){
      if (!opts.draggable) return;
      el.addEventListener('mousedown', function(e){ onDown(e.clientX); });
      window.addEventListener('mousemove', function(e){ onMove(e.clientX); });
      window.addEventListener('mouseup', function(){ onUp(); });
      el.addEventListener('touchstart', function(e){ var t=e.touches[0]; onDown(t.clientX); }, {passive:true});
      el.addEventListener('touchmove', function(e){ var t=e.touches[0]; onMove(t.clientX); }, {passive:true});
      el.addEventListener('touchend', function(){ onUp(); }, {passive:true});
    }

    function load(images, cbFirst, cbAll){
      var loaded = 0; n = (images||[]).length;
      images.forEach(function(url,i){
        var im = new Image();
        im.onload = function(){
          loaded++;
          if (loaded === 1 && cbFirst) cbFirst();
          if (loaded === n && cbAll) cbAll();
        };
        im.src = url;
        imgs[i] = im;
      });
    }

    function resizeAndDraw(){ fit(); draw(); }

    fit();
    load(opts.images || [], function(){ draw(); }, function(){ if (opts.autoplay !== false) play(); });
    bindDrag(canvas);

    return {
      play, pause, toggle, draw, resizeAndDraw,
      isPlaying: function(){ return playing; },
      setPlayingUI: function(btn){
        if (!btn) return;
        function sync(){
          btn.style.backgroundImage = playing ? ICONS.pause : ICONS.play;
          btn.setAttribute('data-mode', playing ? 'pause' : 'play');
        }
        btn.onclick = function(){
          toggle();
          sync();
        };
        sync();
      },
      destroy: function(){ pause(); }
    };
  }

  // -------- MODAL (único) --------
  var modal = document.getElementById('fp360-modal');
  var canvas = document.getElementById('fp360-canvas');
  var ctrlWrap = modal ? modal.querySelector('.fp360-controls') : null;
  var ctrlBtn  = ctrlWrap ? ctrlWrap.querySelector('.fp360-playpause') : null;
  var modalViewer = null;
  var modalResizeHandler = null;

  // ✅ NUEVA FUNCIÓN: Crear modal dinámicamente si no existe
  function ensureModalExists() {
    if (document.getElementById('fp360-modal')) return; // Ya existe
    
    var modalHTML = '<div id="fp360-modal" class="fp360-modal" aria-hidden="true">' +
      '<div class="fp360-modal-backdrop" data-fp360-close></div>' +
      '<div class="fp360-modal-dialog" role="dialog" aria-modal="true" aria-label="visor 360">' +
        '<button class="fp360-modal-close" type="button" data-fp360-close">×</button>' +
        '<div class="fp360-controls" aria-hidden="true">' +
          '<button type="button" class="fp360-ctrl fp360-playpause" data-mode="pause" aria-label="play/pause"></button>' +
        '</div>' +
        '<canvas id="fp360-canvas"></canvas>' +
      '</div>' +
    '</div>';
    
    document.body.insertAdjacentHTML('beforeend', modalHTML);
    
    // Actualizar referencias
    modal = document.getElementById('fp360-modal');
    canvas = document.getElementById('fp360-canvas');
    ctrlWrap = modal ? modal.querySelector('.fp360-controls') : null;
    ctrlBtn  = ctrlWrap ? ctrlWrap.querySelector('.fp360-playpause') : null;
  }

  function openModal(data){
    ensureModalExists(); // ✅ Asegurar que el modal existe antes de abrirlo
    
    document.body.classList.add('fp360-lock');
    modal.setAttribute('aria-hidden','false');
    if (ctrlWrap) ctrlWrap.setAttribute('aria-hidden', data.controls ? 'false' : 'true');

    // clona el botón para limpiar listeners previos
    if (ctrlBtn) {
      var fresh = ctrlBtn.cloneNode(true);
      ctrlBtn.parentNode.replaceChild(fresh, ctrlBtn);
      ctrlBtn = fresh;
    }

    requestAnimationFrame(function(){
      modalViewer = createViewer(canvas, {
        images: data.images || [],
        fps: data.fps || 24,
        controls: !!data.controls,
        draggable: !!data.controls,
        autoplay: true
      });
      if (data.controls && ctrlBtn) modalViewer.setPlayingUI(ctrlBtn);

      modalResizeHandler = function(){
        if (modal.getAttribute('aria-hidden') === 'false' && modalViewer) {
          modalViewer.resizeAndDraw();
        }
      };
      window.addEventListener('resize', modalResizeHandler, { passive:true });
    });
  }

  function closeModal(){
    if (modalViewer) { modalViewer.destroy(); modalViewer = null; }
    if (canvas && canvas.getContext) {
      var ctx = canvas.getContext('2d');
      ctx && ctx.clearRect(0,0,canvas.width,canvas.height);
    }
    if (modalResizeHandler) { window.removeEventListener('resize', modalResizeHandler); modalResizeHandler = null; }
    modal.setAttribute('aria-hidden','true');
    document.body.classList.remove('fp360-lock');
  }

  document.addEventListener('click', function(e){
    var btn = e.target.closest('.fp360-open');
    if (!btn) return;
    var json = btn.getAttribute('data-fp360');
    if (!json) return;
    try { openModal(JSON.parse(json)); } catch(err){}
  });

  document.addEventListener('click', function(e){
    if (e.target.matches('[data-fp360-close]')) closeModal();
  });

  // -------- INLINE (múltiples instancias) --------
  function bootInline(){
    var blocks = document.querySelectorAll('.fp360-inline');
    blocks.forEach(function(block){
      var json = block.getAttribute('data-fp360');
      if (!json) return;
      var data;
      try { data = JSON.parse(json); } catch(e){ return; }
      var cvs = block.querySelector('.fp360-inline-canvas');
      var btn = block.querySelector('.fp360-playpause');
      
      // Marcar como inicializado para ocultar preview
      block.classList.add('fp360-initialized');
      
      var viewer = createViewer(cvs, {
        images: data.images || [],
        fps: data.fps || 24,
        controls: !!data.controls,
        draggable: !!data.controls,
        autoplay: true
      });
      if (data.controls && btn) viewer.setPlayingUI(btn);
      window.addEventListener('resize', function(){ viewer.resizeAndDraw(); }, { passive:true });
    });
  }

  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', bootInline);
  } else {
    bootInline();
  }
})();
